from __future__ import annotations

import os
import sys
import random
import threading
import time
import math
from dataclasses import dataclass, field
from pathlib import Path


def _get_base_path() -> Path:
    """Directory for .txt packets and iball.ico; PyInstaller-friendly."""
    if getattr(sys, "frozen", False):
        return Path(sys.executable).parent
    return Path(__file__).resolve().parent

import tkinter as tk
from tkinter import ttk, messagebox

try:
    import serial  # type: ignore
except Exception:  # pragma: no cover
    serial = None


TICK_MS = 100          # VB6 old_tmr_data_chage.Interval = 100
MAINT_MS = 1000        # VB6 tmr_maintaince.Interval = 1000
ARRIVAL_MS = 100       # VB6 tmrDataArrival.Interval = 100


def _fmt_1dec(x: float) -> str:
    return f"{x:.1f}"


def _safe_float(s: str, default: float = 0.0) -> float:
    try:
        return float(s.strip())
    except Exception:
        return default


def _split_packets(lines: list[str]) -> list[str]:
    # VB6 stored line-by-line in arrays and concatenated all lines.
    # We keep the file as one list of lines with '\r\n' and join at send time.
    return [ln.rstrip("\r\n") for ln in lines]


def _load_packet_file(path: Path) -> list[str]:
    with path.open("r", encoding="utf-8", errors="replace") as f:
        return _split_packets(f.readlines())


def _inject_random_noise(s: str, errors_max: int = 8) -> str:
    # Rough port of VB6 "random noise" injection.
    if len(s) < 10:
        return s
    nerr = int(random.random() * errors_max)
    for _ in range(nerr + 1):
        if len(s) < 10:
            break
        # VB did (Rnd() * Len(tempstr)) - 5; allow a negative and clamp.
        ptr = int(random.random() * len(s)) - 5
        ptr = max(0, min(ptr, len(s)))
        bad = chr(int(random.random() * 254))
        s = s[:ptr] + bad + s[ptr + 1 :]
    return s


@dataclass
class SimState:
    drill_depth: float = 2000.0
    bit_depth: float = 2000.0

    pump1_rate: float = 100.3
    pump2_rate: float = 100.3
    pump3_rate: float = 100.3
    pump1_base: float = 0.0  # chosen random 20–120 when active; rate = base ± 5%
    pump2_base: float = 0.0
    pump3_base: float = 0.0

    # Counters and flags
    timer_reload: int = 30
    seconds_counter: float = 30.0
    inner_seconds_counter: float = 0.0

    send_wits_ok: bool = False
    send_packet: bool = False

    incoming_data: str = ""

    off_bottom_flag: bool = False
    off_bottom_timer: int = 300
    on_bottom_timer: int = 100
    pipe_length: float = 0.0

    random_noise_timer: int = 0
    flash: bool = False

    # 0713 Inclination (deg from horizontal 0–90) / 0715 Azimuth (deg 0–360 from N)
    azimuth_deg: float = 0.0
    inclination_deg: float = 0.0
    target_inclination_deg: float = 0.0
    inclination_build_start_depth: float = 0.0
    inclination_build_length_ft: float = 300.0
    deviation_trigger_depth_ft: float = 5000.0
    deviation_triggered: bool = False
    drill_depth_prev: float = 0.0

    # Packet templates (as list of lines)
    full_lines: list[str] = field(default_factory=list)
    half_lines: list[str] = field(default_factory=list)


class WitsSimApp:
    def __init__(self, root: tk.Tk) -> None:
        self.root = root
        self.root.title("WITS Simulator")

        self.state = SimState()
        # Directory for .txt packets and icon (script dir, or exe dir when built with PyInstaller)
        self.project_dir = _get_base_path()

        # Window icon (place iball.ico in the same folder as the script or .exe)
        try:
            icon_path = self.project_dir / "iball.ico"
            if icon_path.exists():
                self.root.iconbitmap(str(icon_path))
        except Exception:
            pass

        # Style / theme
        self.style = ttk.Style(self.root)
        try:
            # Use a theme that supports color configuration cross-platform.
            self.style.theme_use("clam")
        except Exception:
            pass

        # Serial
        self._ser = None
        self._serial_lock = threading.Lock()

        # UI variables
        self.var_comport = tk.StringVar(value="1")
        self.var_settings = tk.StringVar(value="9600,n,8,1")

        self.var_depth = tk.StringVar(value="2000.00")
        self.var_bit_depth = tk.StringVar(value="2000.00")
        self.var_inclination = tk.StringVar(value="0.0")
        self.var_azimuth = tk.StringVar(value="0.0")

        # Default seconds per automatic WITS packet
        self.var_msg_timer = tk.StringVar(value="2")
        self.var_random_noise = tk.BooleanVar(value=False)

        self.var_dark_mode = tk.BooleanVar(value=True)

        # Displayed pump rates
        self.var_pump1 = tk.StringVar(value="0.0")
        self.var_pump2 = tk.StringVar(value="0.0")
        self.var_pump3 = tk.StringVar(value="0.0")

        self.var_wits_type = tk.StringVar(value="Half")  # Full / Half; default Half WITS

        # Mode: request, auto, epic, totco (VB UI had 4 option buttons)
        # Default to Sim Pason Auto
        self.var_mode = tk.StringVar(value="pason_auto")

        # ROP option maps to timer_reload (VB used integer and decremented each 100ms by 1)
        # Default: 1 Min / Ft
        self.var_rop = tk.StringVar(value="60")  # op_60sec

        # Off-bottom mode
        self.var_offbottom_mode = tk.StringVar(value="every32")  # on, off, 10, 30, 60, every32

        # Pumps mode
        self.var_pumps_mode = tk.StringVar(value="off_on_con")  # on, off, off_on_con

        # Status text
        self.var_last_packet = tk.StringVar(value="Last packet sent --:--:--")
        self.var_connection_info = tk.StringVar(value="")
        self.var_drilling_status = tk.StringVar(value="OFF BOTTOM")
        self.var_comm_status = tk.StringVar(value="Port Inactive")

        # React to off-bottom mode changes like VB click handlers
        self.var_offbottom_mode.trace_add("write", self._on_offbottom_mode_changed)

        self._build_ui()
        self._apply_theme()
        self._load_packets()
        self._apply_rop_from_ui()
        self._on_offbottom_mode_changed()
        self._reset_azimuth_and_inclination()  # random azimuth 0–360, inclination 0, trigger depth 5000±1000
        self.state.drill_depth_prev = _safe_float(self.var_depth.get(), self.state.drill_depth)
        # Start with COM port inactive (no auto-connect)
        self._update_comm_ui()

        # Start timers
        self._schedule_tick()
        self._schedule_maint()
        self._schedule_arrival()

        self.root.protocol("WM_DELETE_WINDOW", self.on_close)

    # ---------------- UI ----------------
    def _build_ui(self) -> None:
        # About ~20% smaller than the original 980x720
        self.root.geometry("800x680")

        top = ttk.Frame(self.root, padding=6)
        top.pack(fill="x")

        ttk.Label(top, text="COM").pack(side="left")
        ttk.Entry(top, width=5, textvariable=self.var_comport).pack(side="left", padx=(5, 8))

        ttk.Label(top, text="Settings").pack(side="left")
        ttk.Entry(top, width=15, textvariable=self.var_settings).pack(side="left", padx=(5, 8))

        # Fixed width so the button doesn't resize when text changes.
        # Width is measured in characters for ttk buttons.
        self.btn_comm = ttk.Button(top, text="COM Port: Inactive", width=26, command=self.toggle_comm)
        self.btn_comm.pack(side="left", padx=(0, 8))
        # (Status text label removed; COM button shows state.)

        # Fixed width so the button doesn't resize when text changes.
        self.btn_start = ttk.Button(top, text="Wits Simulator Stopped", width=24, command=self.toggle_running)
        self.btn_start.pack(side="left", padx=(0, 8))

        self.btn_wits_type = ttk.Button(top, text="Half WITS", command=self.toggle_wits_type)
        self.btn_wits_type.pack(side="left", padx=(0, 8))

        ttk.Button(top, text="Edit WITS", command=self.open_wits_file).pack(side="left")

        mid = ttk.Frame(self.root, padding=(6, 0, 6, 6))
        mid.pack(fill="x")
        mid.columnconfigure(0, weight=1, uniform="midcols")
        mid.columnconfigure(1, weight=1, uniform="midcols")
        mid.columnconfigure(2, weight=1, uniform="midcols")

        # Depth / bit depth
        depth_box = ttk.LabelFrame(mid, text="Depths", padding=6)
        depth_box.grid(row=0, column=0, sticky="nsew", padx=(0, 8))
        depth_box.columnconfigure(0, weight=0)
        depth_box.columnconfigure(1, weight=0)
        depth_box.columnconfigure(2, weight=1)

        ttk.Label(depth_box, text="Current Depth").grid(row=0, column=0, sticky="w")
        e_depth = ttk.Entry(depth_box, textvariable=self.var_depth, width=12)
        e_depth.grid(row=0, column=1, sticky="w", padx=(6, 0))

        ttk.Label(depth_box, text="Bit Depth").grid(row=1, column=0, sticky="w", pady=(6, 0))
        e_bit = ttk.Entry(depth_box, textvariable=self.var_bit_depth, width=12, state="readonly")
        e_bit.grid(row=1, column=1, sticky="w", padx=(6, 0), pady=(6, 0))

        # Visual pacifier as a small circular "clock" showing ROP timer progress
        self.pac_canvas = tk.Canvas(depth_box, width=40, height=40, highlightthickness=0, bd=0)
        self.pac_canvas.grid(row=0, column=2, rowspan=2, sticky="nsw", padx=(8, 0))
        self._init_pacifier_clock()

        ttk.Label(depth_box, text="Inclination (°)").grid(row=2, column=0, sticky="w", pady=(6, 0))
        ttk.Label(depth_box, textvariable=self.var_inclination).grid(row=2, column=1, sticky="w", padx=(6, 0), pady=(6, 0))
        ttk.Label(depth_box, text="Azimuth (°)").grid(row=3, column=0, sticky="w", pady=(4, 0))
        ttk.Label(depth_box, textvariable=self.var_azimuth).grid(row=3, column=1, sticky="w", padx=(6, 0), pady=(4, 0))

        # Drilling status
        self.lbl_status = ttk.Label(depth_box, textvariable=self.var_drilling_status, style="Status.TLabel")
        self.lbl_status.grid(row=4, column=0, columnspan=2, sticky="we", pady=(4, 0))

        ttk.Label(depth_box, textvariable=self.var_connection_info, foreground="red").grid(
            row=5, column=0, columnspan=2, sticky="w", pady=(4, 0)
        )

        # Mode box
        mode_box = ttk.LabelFrame(mid, text="EDR Simulation Mode", padding=6)
        mode_box.grid(row=0, column=1, sticky="nsew", padx=(0, 8))

        ttk.Radiobutton(mode_box, text="Sim Pason Request Mode", value="pason_req", variable=self.var_mode).pack(
            anchor="w"
        )
        ttk.Radiobutton(mode_box, text="Sim Pason Auto", value="pason_auto", variable=self.var_mode).pack(anchor="w")
        ttk.Radiobutton(mode_box, text="Sim EPIC", value="epic", variable=self.var_mode).pack(anchor="w")
        ttk.Radiobutton(mode_box, text="Sim TOTCO", value="totco", variable=self.var_mode).pack(anchor="w")

        msgf = ttk.Frame(mode_box)
        msgf.pack(fill="x", pady=(6, 0))
        ttk.Label(msgf, text="Seconds per automatic WITS").pack(side="left")
        ttk.Entry(msgf, width=5, textvariable=self.var_msg_timer).pack(side="left", padx=(6, 0))

        ttk.Checkbutton(mode_box, text="Add Random Noise", variable=self.var_random_noise).pack(anchor="w", pady=(6, 0))
        ttk.Label(mode_box, textvariable=self.var_last_packet, font=("Segoe UI", 9, "bold")).pack(
            anchor="w", pady=(8, 0)
        )

        # ROP box
        rop_box = ttk.LabelFrame(mid, text="Rate Of Penetration", padding=6)
        rop_box.grid(row=0, column=2, sticky="nsew")

        rops = [
            ("ROP 10 Min / Ft", "600"),
            ("ROP 5 Min / Ft", "300"),
            ("ROP 2 Min / Ft", "120"),
            ("ROP 1 Min / Ft", "60"),
            ("ROP 10 Sec / Ft", "10"),
            ("ROP 5 Sec / Ft", "5"),
            ("ROP 2 Sec / Ft", "2"),
            ("ROP 1 Sec / Ft", "1"),
        ]
        for txt, val in rops:
            ttk.Radiobutton(rop_box, text=txt, value=val, variable=self.var_rop, command=self._apply_rop_from_ui).pack(
                anchor="w"
            )

        # Bottom row - pumps + off-bottom + text windows
        bottom = ttk.Frame(self.root, padding=6)
        bottom.pack(fill="both", expand=True)

        leftcol = ttk.Frame(bottom)
        leftcol.pack(side="left", fill="y", padx=(0, 8))

        pumps = ttk.LabelFrame(leftcol, text="Pumps", padding=6)
        pumps.pack(fill="x")
        ttk.Radiobutton(pumps, text="Pumps on", value="on", variable=self.var_pumps_mode).pack(anchor="w")
        ttk.Radiobutton(pumps, text="Pumps off", value="off", variable=self.var_pumps_mode).pack(anchor="w")
        ttk.Radiobutton(
            pumps, text="Pumps off When Off Bottom", value="off_on_con", variable=self.var_pumps_mode
        ).pack(anchor="w")

        # Current pump rates
        pump_vals = ttk.Frame(pumps)
        pump_vals.pack(fill="x", pady=(4, 0))
        ttk.Label(pump_vals, text="Pump 1:").grid(row=0, column=0, sticky="w")
        ttk.Label(pump_vals, textvariable=self.var_pump1).grid(row=0, column=1, sticky="w", padx=(4, 0))
        ttk.Label(pump_vals, text="Pump 2:").grid(row=1, column=0, sticky="w")
        ttk.Label(pump_vals, textvariable=self.var_pump2).grid(row=1, column=1, sticky="w", padx=(4, 0))
        ttk.Label(pump_vals, text="Pump 3:").grid(row=2, column=0, sticky="w")
        ttk.Label(pump_vals, textvariable=self.var_pump3).grid(row=2, column=1, sticky="w", padx=(4, 0))

        offb = ttk.LabelFrame(leftcol, text="Off Bottom Bit Control", padding=6)
        offb.pack(fill="x", pady=(8, 0))
        ttk.Radiobutton(offb, text="On Bottom", value="on", variable=self.var_offbottom_mode).pack(anchor="w")
        ttk.Radiobutton(offb, text="Off Bottom", value="off", variable=self.var_offbottom_mode).pack(anchor="w")
        ttk.Radiobutton(offb, text="Off Bottom Every 10min for 3 min", value="10", variable=self.var_offbottom_mode).pack(
            anchor="w"
        )
        ttk.Radiobutton(offb, text="Off Bottom Every 30min for 3 min", value="30", variable=self.var_offbottom_mode).pack(
            anchor="w"
        )
        ttk.Radiobutton(offb, text="Off Bottom Every 60min for 3 min", value="60", variable=self.var_offbottom_mode).pack(
            anchor="w"
        )
        ttk.Radiobutton(
            offb, text="Off Bottom Every 32 ft for 5 min", value="every32", variable=self.var_offbottom_mode
        ).pack(anchor="w")

        timers = ttk.Frame(offb)
        timers.pack(fill="x", pady=(6, 0))
        self.var_on_bot = tk.StringVar(value="100")
        self.var_off_bot = tk.StringVar(value="0")
        ttk.Label(timers, text="On Btm Tmr Sec").grid(row=0, column=0, sticky="w")
        ttk.Entry(timers, width=8, textvariable=self.var_on_bot, state="readonly").grid(row=0, column=1, padx=(6, 0))
        ttk.Label(timers, text="Off Btm Tmr Sec").grid(row=1, column=0, sticky="w", pady=(6, 0))
        ttk.Entry(timers, width=8, textvariable=self.var_off_bot, state="readonly").grid(row=1, column=1, padx=(6, 0), pady=(6, 0))

        # Bottom-left window control
        theme_box = ttk.Frame(leftcol)
        theme_box.pack(side="bottom", fill="x", pady=(8, 0))
        ttk.Checkbutton(theme_box, text="Dark mode", variable=self.var_dark_mode, command=self._apply_theme).pack(
            anchor="w"
        )

        # Text windows (send/receive)
        rightcol = ttk.Frame(bottom)
        rightcol.pack(side="left", fill="both", expand=True)

        # Sending pane: text + scrollbar in one row so scrollbar is attached
        ttk.Label(rightcol, text="Sending To Rigwire").pack(anchor="center")
        send_frame = ttk.Frame(rightcol)
        send_frame.pack(fill="both", expand=True)
        send_frame.columnconfigure(0, weight=1)
        send_frame.rowconfigure(0, weight=1)
        self.txt_send = tk.Text(
            send_frame,
            height=8,
            bg="black",
            fg="yellow",
            wrap="none",
            state="disabled",
            font=("Consolas", 8),
        )
        self.txt_send.grid(row=0, column=0, sticky="nsew")
        self.txt_send.tag_configure("center", justify="center")
        send_scroll = tk.Scrollbar(send_frame, orient="vertical", command=self.txt_send.yview)
        send_scroll.grid(row=0, column=1, sticky="ns")
        self.txt_send.configure(yscrollcommand=send_scroll.set)

        # Receiving pane: text + scrollbar in one row so scrollbar is attached
        ttk.Label(rightcol, text="Receiving From Rigwire").pack(anchor="center", pady=(6, 0))
        recv_frame = ttk.Frame(rightcol)
        recv_frame.pack(fill="both", expand=True)
        recv_frame.columnconfigure(0, weight=1)
        recv_frame.rowconfigure(0, weight=1)
        self.txt_recv = tk.Text(
            recv_frame,
            height=8,
            bg="black",
            fg="yellow",
            wrap="none",
            state="disabled",
            font=("Consolas", 8),
        )
        self.txt_recv.grid(row=0, column=0, sticky="nsew")
        self.txt_recv.tag_configure("center", justify="center")
        recv_scroll = tk.Scrollbar(recv_frame, orient="vertical", command=self.txt_recv.yview)
        recv_scroll.grid(row=0, column=1, sticky="ns")
        self.txt_recv.configure(yscrollcommand=recv_scroll.set)

    def _init_pacifier_clock(self) -> None:
        # Prepare a simple circular clock: outer ring + hand
        size = 32
        margin = 4
        self.pac_center = margin + size / 2.0
        self.pac_radius = size / 2.0
        self.pac_canvas.configure(width=size + margin * 2, height=size + margin * 2)

        # Background circle and initial hand pointing up
        self.pac_bg_circle = self.pac_canvas.create_oval(
            margin,
            margin,
            margin + size,
            margin + size,
            width=2,
            outline="",
        )
        self.pac_hand = self.pac_canvas.create_line(
            self.pac_center,
            self.pac_center,
            self.pac_center,
            margin,
            width=2,
            capstyle=tk.ROUND,
        )

    def _update_pacifier_clock(self, frac: float) -> None:
        # frac is 0.0–1.0, representing progress around the circle
        frac = max(0.0, min(1.0, frac))
        angle_deg = -90.0 + 360.0 * frac  # start pointing up
        angle_rad = math.radians(angle_deg)
        r = self.pac_radius * 0.8
        x = self.pac_center + r * math.cos(angle_rad)
        y = self.pac_center + r * math.sin(angle_rad)
        self.pac_canvas.coords(self.pac_hand, self.pac_center, self.pac_center, x, y)

    def _apply_theme(self) -> None:
        dark = bool(self.var_dark_mode.get())

        if dark:
            bg = "#1e1e1e"
            fg = "#e6e6e6"
            muted = "#b9b9b9"
            entry_bg = "#2a2a2a"
            text_bg = "#0f0f0f"
            text_fg = "#ffff66"  # soft yellow in dark mode
            btn_bg = "#2d2d2d"
            btn_active = "#3a3a3a"
            danger_bg = "#7a1f1f"
            warn_bg = "#7a5a1f"
            ok_bg = "#1f7a3a"
            off_bg = "#d27d2c"  # orange-ish for OFF BOTTOM
        else:
            bg = "#f0f0f0"
            fg = "#111111"
            muted = "#444444"
            entry_bg = "#ffffff"
            text_bg = "#ffffff"
            text_fg = "#aa8800"  # darker yellow/brown for light mode
            btn_bg = "#e6e6e6"
            btn_active = "#dcdcdc"
            danger_bg = "#e06c75"
            warn_bg = "#e5c07b"
            ok_bg = "#98c379"
            off_bg = "#ffb347"

        self.root.configure(background=bg)

        # Base widget styling
        self.style.configure(".", background=bg, foreground=fg)
        self.style.configure("TFrame", background=bg)
        self.style.configure("TLabel", background=bg, foreground=fg)
        self.style.configure("TButton", background=btn_bg, foreground=fg, padding=(10, 6))
        self.style.map("TButton", background=[("active", btn_active), ("pressed", btn_active)])

        # LabelFrames (tk uses 'TLabelframe')
        self.style.configure("TLabelframe", background=bg, foreground=fg)
        self.style.configure("TLabelframe.Label", background=bg, foreground=fg)

        # Inputs
        self.style.configure("TEntry", fieldbackground=entry_bg, foreground=fg, background=bg, insertcolor=fg)
        self.style.configure("TCombobox", fieldbackground=entry_bg, foreground=fg, background=bg, arrowcolor=fg)
        self.style.map(
            "TCombobox",
            fieldbackground=[("readonly", entry_bg)],
            foreground=[("readonly", fg)],
        )

        self.style.configure("TRadiobutton", background=bg, foreground=fg)
        self.style.configure("TCheckbutton", background=bg, foreground=fg)

        # Accent button states used by the simulator flashing logic
        self.style.configure("Ok.TButton", background=ok_bg, foreground="black")
        self.style.map("Ok.TButton", background=[("active", ok_bg)])
        self.style.configure("Warn.TButton", background=warn_bg, foreground="black")
        self.style.map("Warn.TButton", background=[("active", warn_bg)])
        self.style.configure("Danger.TButton", background=danger_bg, foreground="black")
        self.style.map("Danger.TButton", background=[("active", danger_bg)])

        # Drilling status label styles
        self.style.configure("Status.TLabel", background=bg, foreground=fg, anchor="center")
        self.style.configure("OnBottom.TLabel", background=ok_bg, foreground="black", anchor="center")
        self.style.configure("OffBottom.TLabel", background=off_bg, foreground="black", anchor="center")
        # Text windows and pacifier canvas (tk widgets, not ttk)
        try:
            # Send/recv boxes always black background in light and dark mode
            self.txt_send.configure(bg="black", fg=text_fg, insertbackground=text_fg)
            self.txt_recv.configure(bg="black", fg=text_fg, insertbackground=text_fg)
            self.pac_canvas.configure(bg=bg, highlightbackground=bg)
            # Use status colors for the clock, with a yellow hand
            self.pac_canvas.itemconfigure(self.pac_bg_circle, outline=fg)
            self.pac_canvas.itemconfigure(self.pac_hand, fill=text_fg)
        except Exception:
            pass

    # ---------------- Packet handling ----------------
    def _load_packets(self) -> None:
        # On startup, search for packet .txt files in the same directory as the main program.
        candidates_full = [self.project_dir / "Full_Wits_Packet.txt", ]
        candidates_half = [self.project_dir / "Half_Wits_Packet.txt", ]

        full_path = next((p for p in candidates_full if p.exists()), None)
        half_path = next((p for p in candidates_half if p.exists()), None)

        if not full_path or not half_path:
            messagebox.showwarning(
                "Missing packet files",
                "Could not find the WITS packet template files next to wits_sim.pyw.\n"
                "Expected Full_Wits_Packet.txt and Half_Wits_Packet.txt.\n"
                "Current File path: {}".format(self.project_dir),
            )
            return
        self.state.full_lines = _load_packet_file(full_path)
        self.state.half_lines = _load_packet_file(half_path)

    def _build_packet_string(self) -> str:
        # Select template
        lines = self.state.full_lines if self.var_wits_type.get() == "Full" else self.state.half_lines

        drill = _safe_float(self.var_depth.get(), self.state.drill_depth)
        self.state.drill_depth = drill
        self.state.bit_depth = _safe_float(self.var_bit_depth.get(), self.state.bit_depth)

        out_lines: list[str] = []
        for ln in lines:
            if len(ln) > 4:
                code = ln[:4]
                if code == "0108":
                    ln = "0108" + _fmt_1dec(self.state.bit_depth)
                elif code == "0110":
                    ln = "0110" + _fmt_1dec(self.state.drill_depth)
                elif code == "1108":
                    ln = "1108" + _fmt_1dec(self.state.drill_depth)
                elif code == "0123":
                    ln = "0123" + str(self.state.pump1_rate)
                elif code == "0124":
                    ln = "0124" + str(self.state.pump2_rate)
                elif code == "0125":
                    ln = "0125" + str(self.state.pump3_rate)
                elif code == "0713":
                    ln = "0713" + f"{self.state.inclination_deg:.1f}"
                elif code == "0715":
                    ln = "0715" + f"{self.state.azimuth_deg:.1f}"
            out_lines.append(ln)

        temp = "\r\n".join(out_lines) + "\r\n"
        if self.var_random_noise.get() and self.state.random_noise_timer == 0:
            self.state.random_noise_timer = int(random.random() * 15)
            temp = _inject_random_noise(temp)
        return temp

    # ---------------- Serial ----------------
    def _open_serial(self) -> None:
        if serial is None:
            raise RuntimeError("pyserial not installed.")
        port = f"COM{int(_safe_float(self.var_comport.get(), 1))}"
        settings = self.var_settings.get().strip()
        # Expected "baud,parity,data,stop" like "9600,n,8,1"
        parts = [p.strip() for p in settings.split(",") if p.strip()]
        baud = int(parts[0]) if parts else 9600
        parity = "N"
        bytesize = 8
        stopbits = 1
        if len(parts) >= 2:
            parity = parts[1].upper()[0]
        if len(parts) >= 3:
            bytesize = int(parts[2])
        if len(parts) >= 4:
            stopbits = float(parts[3])

        par_map = {
            "N": serial.PARITY_NONE,
            "E": serial.PARITY_EVEN,
            "O": serial.PARITY_ODD,
            "M": serial.PARITY_MARK,
            "S": serial.PARITY_SPACE,
        }
        bs_map = {5: serial.FIVEBITS, 6: serial.SIXBITS, 7: serial.SEVENBITS, 8: serial.EIGHTBITS}
        sb_map = {1: serial.STOPBITS_ONE, 1.5: serial.STOPBITS_ONE_POINT_FIVE, 2: serial.STOPBITS_TWO}

        self._ser = serial.Serial(
            port=port,
            baudrate=baud,
            parity=par_map.get(parity, serial.PARITY_NONE),
            bytesize=bs_map.get(bytesize, serial.EIGHTBITS),
            stopbits=sb_map.get(stopbits, serial.STOPBITS_ONE),
            timeout=0,
            write_timeout=0,
        )

    def _close_serial(self) -> None:
        if self._ser is not None:
            try:
                self._ser.close()
            except Exception:
                pass
        self._ser = None

    def toggle_comm(self) -> None:
        with self._serial_lock:
            if self._ser is None:
                try:
                    self._open_serial()
                except Exception as e:
                    self._ser = None
                    self.var_comm_status.set("Port error")
                    self._append_send(f"The comm port is being used by another app. ({e})\r\n")
                    self._update_comm_ui()
                    return
                self.var_comm_status.set(f"Port COM{self.var_comport.get()} Active")
            else:
                self._close_serial()
                self.var_comm_status.set("Port Inactive")
        self._update_comm_ui()

    def _attempt_autoconnect(self) -> None:
        # VB6 Form_Load attempts to open the port immediately.
        try:
            self.toggle_comm()
        except Exception:
            pass

    def _update_comm_ui(self) -> None:
        active = self._ser is not None
        if active:
            self.btn_comm.configure(text=f"COM Port: COM{self.var_comport.get()} Active", style="Ok.TButton")
        else:
            self.btn_comm.configure(text="COM Port: Inactive", style="Danger.TButton")

    def _serial_write(self, data: str) -> None:
        with self._serial_lock:
            if self._ser is None:
                return
            try:
                self._ser.write(data.encode("latin-1", errors="replace"))
            except Exception as e:
                self._append_send(f"Send WITS Error: {e}\r\n")

    def _serial_read_available(self) -> str:
        with self._serial_lock:
            if self._ser is None:
                return ""
            try:
                n = self._ser.in_waiting
                if not n:
                    return ""
                b = self._ser.read(n)
                return b.decode("latin-1", errors="replace")
            except Exception:
                return ""

    # ---------------- Actions ----------------
    def toggle_running(self) -> None:
        self.state.send_wits_ok = not self.state.send_wits_ok
        if self.state.send_wits_ok:
            self.btn_start.configure(text="Wits Simulator Running")
        else:
            self.btn_start.configure(text="Wits Simulator Stopped")

    def toggle_wits_type(self) -> None:
        if self.var_wits_type.get() == "Full":
            self.var_wits_type.set("Half")
            self.btn_wits_type.configure(text="Half WITS")
        else:
            self.var_wits_type.set("Full")
            self.btn_wits_type.configure(text="Full WITS")

    def open_wits_file(self) -> None:
        try:
            name = "Full_Wits_Packet.txt" if self.var_wits_type.get() == "Full" else "Half_Wits_Packet.txt"
            p = self.project_dir / name
            if not p.exists():
                messagebox.showinfo("Not found", f"Could not find {p}")
                return
            os.startfile(str(p))  # type: ignore[attr-defined]
        except Exception as e:
            messagebox.showinfo("Error", f"Error opening WITS file: {e}")

    # ---------------- Timers / logic ----------------
    def _apply_rop_from_ui(self) -> None:
        try:
            self.state.timer_reload = int(self.var_rop.get())
        except Exception:
            self.state.timer_reload = 30
        self.state.seconds_counter = float(self.state.timer_reload)

    def _schedule_tick(self) -> None:
        self.root.after(TICK_MS, self._tick_100ms)

    def _schedule_maint(self) -> None:
        self.root.after(MAINT_MS, self._tick_maint)

    def _schedule_arrival(self) -> None:
        self.root.after(ARRIVAL_MS, self._tick_arrival)

    def _on_offbottom_mode_changed(self, *_: object) -> None:
        self._update_offbottom_mode()

        drill = _safe_float(self.var_depth.get(), self.state.drill_depth)
        self.state.drill_depth = drill

        mode = self.var_offbottom_mode.get()
        if mode == "on":
            self.state.bit_depth = drill
        elif mode in ("off", "10", "30", "60"):
            self.state.bit_depth = drill - 36
        else:
            # every32: keep current relationship based on flag
            self.state.bit_depth = drill - 36 if self.state.off_bottom_flag else drill

        self.var_depth.set(f"{self.state.drill_depth:0.2f}")
        self.var_bit_depth.set(f"{self.state.bit_depth:0.2f}")

    def _reset_azimuth_and_inclination(self) -> None:
        """Random azimuth 0–360°, inclination 0°, new deviation trigger depth 5000±1000 ft."""
        self.state.azimuth_deg = random.uniform(0.0, 360.0)
        self.state.inclination_deg = 0.0
        self.state.target_inclination_deg = 0.0
        self.state.inclination_build_start_depth = 0.0
        self.state.inclination_build_length_ft = random.uniform(200.0, 400.0)
        self.state.deviation_trigger_depth_ft = 5000.0 + random.uniform(-1000.0, 1000.0)
        self.state.deviation_triggered = False
        self.var_inclination.set(f"{self.state.inclination_deg:.1f}")
        self.var_azimuth.set(f"{self.state.azimuth_deg:.1f}")

    def _update_inclination_azimuth(self, drill_depth: float) -> None:
        """Update 0713 inclination and 0715 azimuth from depth (call when on bottom and depth changes)."""
        # Reset when drill depth is below 1000 ft
        if drill_depth < 1000.0 and self.state.drill_depth_prev >= 1000.0:
            self._reset_azimuth_and_inclination()
        self.state.drill_depth_prev = drill_depth

        if drill_depth < 1000.0:
            return

        # Start deviation at 5000 ± 1000 ft
        if not self.state.deviation_triggered and drill_depth >= self.state.deviation_trigger_depth_ft:
            self.state.deviation_triggered = True
            self.state.target_inclination_deg = random.uniform(0.0, 90.0)
            self.state.inclination_build_start_depth = drill_depth
            self.state.inclination_build_length_ft = random.uniform(200.0, 400.0)

        if self.state.deviation_triggered:
            build_end = self.state.inclination_build_start_depth + self.state.inclination_build_length_ft
            if drill_depth <= build_end:
                frac = (drill_depth - self.state.inclination_build_start_depth) / self.state.inclination_build_length_ft
                self.state.inclination_deg = frac * self.state.target_inclination_deg
            else:
                self.state.inclination_deg = self.state.target_inclination_deg

        self.var_inclination.set(f"{self.state.inclination_deg:.1f}")
        self.var_azimuth.set(f"{self.state.azimuth_deg:.1f}")

    def _update_offbottom_mode(self) -> None:
        mode = self.var_offbottom_mode.get()
        if mode == "on":
            self.state.off_bottom_timer = 180
            self.state.on_bottom_timer = 600
            self.state.off_bottom_flag = False
        elif mode in ("off", "10", "30", "60"):
            self.state.off_bottom_timer = 180
            self.state.on_bottom_timer = 600 if mode == "10" else 1800 if mode == "30" else 3600 if mode == "60" else 600
            self.state.off_bottom_flag = True
        elif mode == "every32":
            self.state.off_bottom_timer = 300
            # On-bottom timer is not used for the every-32ft mode; keep a
            # cosmetic default so the UI shows 100 on startup.
            self.state.on_bottom_timer = 100
            self.state.pipe_length = 0.0

    def _tick_maint(self) -> None:
        try:
            if self.state.random_noise_timer > 0:
                self.state.random_noise_timer -= 1

            # Flash start button if not running (rough VB behavior)
            if not self.state.send_wits_ok:
                self.state.flash = not self.state.flash
                if self.state.flash:
                    self.btn_start.configure(style="Danger.TButton")
                else:
                    self.btn_start.configure(style="Warn.TButton")
            else:
                self.btn_start.configure(style="Ok.TButton")

            # Pumps logic: when active, rate = chosen base (20–120) ± 5% each tick
            pumps = self.var_pumps_mode.get()
            if pumps == "off_on_con":
                if int(self.state.drill_depth) != int(self.state.bit_depth):
                    self.state.pump1_rate = self.state.pump2_rate = self.state.pump3_rate = 0.0
                    self.state.pump1_base = self.state.pump2_base = self.state.pump3_base = 0.0
                else:
                    if self.state.pump1_base == 0.0:
                        self.state.pump1_base = random.uniform(20.0, 120.0)
                        self.state.pump2_base = random.uniform(20.0, 120.0)
                        self.state.pump3_base = random.uniform(20.0, 120.0)
                    self.state.pump1_rate = self.state.pump1_base * random.uniform(0.95, 1.05)
                    self.state.pump2_rate = self.state.pump2_base * random.uniform(0.95, 1.05)
                    self.state.pump3_rate = self.state.pump3_base * random.uniform(0.95, 1.05)
            elif pumps == "on":
                if self.state.pump1_base == 0.0:
                    self.state.pump1_base = random.uniform(20.0, 120.0)
                    self.state.pump2_base = random.uniform(20.0, 120.0)
                    self.state.pump3_base = random.uniform(20.0, 120.0)
                self.state.pump1_rate = self.state.pump1_base * random.uniform(0.95, 1.05)
                self.state.pump2_rate = self.state.pump2_base * random.uniform(0.95, 1.05)
                self.state.pump3_rate = self.state.pump3_base * random.uniform(0.95, 1.05)
            elif pumps == "off":
                self.state.pump1_rate = self.state.pump2_rate = self.state.pump3_rate = 0.0
                self.state.pump1_base = self.state.pump2_base = self.state.pump3_base = 0.0

            # Reflect pump rates in UI
            self.var_pump1.set(f"{self.state.pump1_rate:0.1f}")
            self.var_pump2.set(f"{self.state.pump2_rate:0.1f}")
            self.var_pump3.set(f"{self.state.pump3_rate:0.1f}")

            # Update off-bottom cycling timers (10/30/60/every32)
            mode = self.var_offbottom_mode.get()
            if mode in ("10", "30", "60"):
                if self.state.off_bottom_flag:
                    self.state.off_bottom_timer = max(0, self.state.off_bottom_timer - 1)
                    if self.state.off_bottom_timer == 0:
                        self.state.off_bottom_timer = 180
                        self.state.off_bottom_flag = False
                else:
                    self.state.on_bottom_timer = max(0, self.state.on_bottom_timer - 1)
                    if self.state.on_bottom_timer == 0:
                        self.state.on_bottom_timer = 600 if mode == "10" else 1800 if mode == "30" else 3600
                        self.state.off_bottom_flag = True

            if mode == "every32" and self.state.off_bottom_flag:
                self.state.off_bottom_timer = max(0, self.state.off_bottom_timer - 1)
                if self.state.off_bottom_timer == 0:
                    self.state.off_bottom_timer = 600
                    self.state.off_bottom_flag = False

            self.var_on_bot.set(str(self.state.on_bottom_timer))
            self.var_off_bot.set(str(self.state.off_bottom_timer))
        except Exception as e:
            self._append_send(f"Maintainance timer Err: {e}\r\n")
        finally:
            self._schedule_maint()

    def _tick_100ms(self) -> None:
        try:
            # Update mode-derived state for on/off bottom selection
            if self.var_offbottom_mode.get() in ("on", "off", "10", "30", "60", "every32"):
                # only re-init on direct on/off clicks would be closer, but ok to keep stable:
                pass

            if not self.state.off_bottom_flag:
                self.var_drilling_status.set("ON BOTTOM")
                self.lbl_status.configure(style="OnBottom.TLabel")
            else:
                self.var_drilling_status.set("OFF BOTTOM")
                self.lbl_status.configure(style="OffBottom.TLabel")

            # Connection info every 32 ft
            if self.var_offbottom_mode.get() == "every32":
                self.var_connection_info.set(f"Connection in {32.0 - self.state.pipe_length:0.2f} ft")
                if self.state.pipe_length >= 32.0:
                    self.state.pipe_length = 0.0
                    self.state.off_bottom_timer = 300
                    self.state.off_bottom_flag = True
            else:
                self.var_connection_info.set("")

            # ROP / depth changing (VB decremented by 1 each 100ms tick)
            if self.state.seconds_counter > 0:
                self.state.seconds_counter -= 1
            if self.state.seconds_counter <= 0:
                self.state.seconds_counter = float(self.state.timer_reload)
                if not self.state.off_bottom_flag:
                    self.state.drill_depth = _safe_float(self.var_depth.get(), self.state.drill_depth) + 0.1
                    self.state.pipe_length += 0.1
                    self.var_depth.set(f"{self.state.drill_depth:0.1f}")
                    self.state.bit_depth = self.state.drill_depth
                    self.var_bit_depth.set(f"{self.state.bit_depth:0.1f}")
                else:
                    self.state.drill_depth = _safe_float(self.var_depth.get(), self.state.drill_depth)
                    self.state.bit_depth = self.state.drill_depth - 36
                    self.var_bit_depth.set(f"{self.state.bit_depth:0.1f}")

                self._update_inclination_azimuth(self.state.drill_depth)

            # Update pacifier "clock" to show progress of timer reload when running
            try:
                if self.state.send_wits_ok and self.state.timer_reload > 0:
                    elapsed = float(self.state.timer_reload) - float(self.state.seconds_counter)
                    frac = max(0.0, min(1.0, elapsed / float(self.state.timer_reload)))
                    self._update_pacifier_clock(frac)
                else:
                    self._update_pacifier_clock(0.0)
            except Exception:
                pass

            # Serial data only goes out every x seconds (x = Seconds per automatic WITS)
            self.state.inner_seconds_counter += 0.1
            x_seconds = _safe_float(self.var_msg_timer.get(), 1.0)
            if self.state.send_wits_ok and self.state.inner_seconds_counter >= x_seconds:
                self.state.inner_seconds_counter = 0.0
                self.send_wits_packet()
        except Exception as e:
            self._append_send(f"Data Change Timer Err: {e}\r\n")
        finally:
            self._schedule_tick()

    def _tick_arrival(self) -> None:
        try:
            s = self._serial_read_available()
            if s:
                self._append_recv(s)
                # Sends are only on the timer (Seconds per automatic WITS), not on "!!"
        except Exception as e:
            self._append_send(f"Timer Data Arrival Err: {e}\r\n")
        finally:
            self._schedule_arrival()

    def send_wits_packet(self) -> None:
        pkt = self._build_packet_string()
        self.var_last_packet.set(time.strftime("Last packet sent %H:%M:%S"))
        # Always show what we attempt to send
        self._append_send(pkt)
        # Also send out the serial port when available
        if self._ser is not None:
            self._serial_write(pkt)

    # ---------------- Text helpers ----------------
    def _append_send(self, s: str) -> None:
        # Temporarily enable for programmatic updates
        self.txt_send.configure(state="normal")
        self.txt_send.insert("end", s, "center")
        # Limit buffer to ~64KB
        content = self.txt_send.get("1.0", "end-1c")
        max_bytes = 65536
        if len(content) > max_bytes:
            # Keep only the last max_bytes characters
            start_index = f"1.0+{len(content) - max_bytes}c"
            self.txt_send.delete("1.0", start_index)
        self.txt_send.see("end")
        self.txt_send.configure(state="disabled")

    def _append_recv(self, s: str) -> None:
        # Temporarily enable for programmatic updates
        self.txt_recv.configure(state="normal")
        self.txt_recv.insert("end", s, "center")
        # Limit buffer to ~64KB
        content = self.txt_recv.get("1.0", "end-1c")
        max_bytes = 65536
        if len(content) > max_bytes:
            start_index = f"1.0+{len(content) - max_bytes}c"
            self.txt_recv.delete("1.0", start_index)
        self.txt_recv.see("end")
        self.txt_recv.configure(state="disabled")

    # ---------------- Shutdown ----------------
    def on_close(self) -> None:
        try:
            self._close_serial()
        finally:
            self.root.destroy()


def main() -> None:
    root = tk.Tk()
    WitsSimApp(root)
    root.mainloop()


if __name__ == "__main__":
    main()
